/*
 * Manage FMS settings-
 * fm_settings
 * 	-p <parameter> <value> - set parameter to <value>
 * 	-l - list all parameter values
 */

#include <unistd.h>
#include <stdio.h>

#include "db.h"
#include "libfma.h"
#include "lf_dflt_error.h"
#include "lf_fms_settings.h"

/*
 * Use preprocessor hackery to have only one list of parameters
 */
#define LF_FMS_ONE_SETTING(name, decl, type, dflt, sunits, punits) \
    { #name, type, #dflt, #sunits, #punits },

#include "lf_fms_settings_def.h"

static struct lf_fms_param_def Params[] = {
  LF_FMS_ALL_SETTINGS
  { NULL, LF_FPT_COUNT, NULL, NULL, NULL }
};

/*
 * prototypes
 */
static void print_all_settings_from_fms(void);
static void set_param_in_fms(char *fms_server, char *param, char *value);
static void print_all_settings_from_db(char *fms_run, char *db_name);
static void set_param_in_db(char *fms_run, char *db_name,
  char *param, char *value);
static void validate_param(char *param, char *value);
static void prepare_param_printing(void);
static void print_param(struct lf_fms_param_def *pp, char *value, int is_dflt);

static void
usage()
{
  fprintf(stderr, "This is used to view and change settings for the FMS.  This should\n");
  fprintf(stderr, "be run to change parameters while the FMS is not running, and the\n");
  fprintf(stderr, "FMS will pick up the new values when it restarts.\n\n");
  fprintf(stderr, "Usage: fm_settings\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -p <paramater> <value> - set <parameter> to <value>\n");
  fprintf(stderr, "  -l - list all parameters and their values\n");
  fprintf(stderr, "  -R <fms_run> - specify FMS_RUN directory\n");
  fprintf(stderr, "  -N <fms_db_name> - specify database name\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

int
main(
  int argc,
  char **argv)
{
  int c;
  extern char *optarg;
  extern int optind;
  char *fms_server;
  char *fms_run;
  char *db_name;
  char *param;
  char *value;
  int db_direct;
  int print_all;

  lf_init();

  /* database table not open yet */
  print_all = FALSE;
  param = NULL;
  db_direct = TRUE;	/* XXX for now */
  value = NULL;

  /* start with default FMS server */
  fms_server = getenv(LF_ENV_FMS_SERVER);
  if (fms_server == NULL || *fms_server == '\0') {
    fms_server = Lf_dflt_fms_server;
  }

  /* start with default run directory */
  fms_run = getenv(LF_ENV_FMS_RUN);
  if (fms_run == NULL || *fms_run == '\0') fms_run = LF_DFLT_FMS_RUN;

  /* start with default database name */
  db_name = getenv(LF_ENV_DB_NAME);
  if (db_name == NULL || *db_name == '\0') db_name = LF_DFLT_DB_NAME;

  /*
   * Parse command line args
   */
  while ((c = getopt(argc, argv, "hp:lR:N:ds:V")) != EOF) switch (c) {

    case 'V':
      printf("FMS version is %s\n", Lf_version);
      exit(0);
      break;

    case 'h':
      usage();
      break;

    case 'l':
      print_all = TRUE;
      break;

    case 'p':
      if (param != NULL) {
	fprintf(stderr, "Only one parameter at a time, please.\n");
	exit(1);
      }

      param = optarg;
      if (optind < argc) {
	value = argv[optind];
	++optind;
      } else {
	fprintf(stderr, "Error: missing value for \"%s\"\n", param);
	exit(1);
      }
      break;

    case 's':
      fms_server = optarg;
      break;

    case 'd':
      db_direct = TRUE;		/* go direct to the database */
      break;

    case 'R':	/* wiring list file (output) */
      fms_run = optarg;
      break;

    case 'N':  /* Host port Id to test out of */
      db_name = optarg;
      break;
  }

  /*
   * If a parameter specified, process it
   */
  if (param != NULL) {

    /* Make sure this is a valid parameter/value pair */
    validate_param(param, value);

    /*
     * either set this just in the database, or else send a message
     * to the FMS to set it
     */
    if (db_direct) {
      set_param_in_db(fms_run, db_name, param, value);

    } else {
      set_param_in_fms(fms_server, param, value);
    }
  }

  /*
   * If full print requested, do it now
   */
  if (print_all) {
    if (db_direct) {
      print_all_settings_from_db(fms_run, db_name);
    } else {
      print_all_settings_from_fms();
    }
  }

  exit(0);
}

/*
 * Set this parameter value in the settings database
 */
static void
set_param_in_db(
  char *fms_run,
  char *db_name,
  char *param,
  char *value)
{
  db_database_ptr_t dbp;
  db_table_ptr_t tp;
  db_datum_t *row;
  int *colids;
  int ncol;
  int rc;

  /*
   * open the database table, creating it if it does not exist
   */
  dbp = db_open_database(fms_run, db_name, TRUE);
  if (dbp == NULL) LF_ERROR(("Error opening database"));

  /* open/create the table */
  tp = lf_open_fms_settings_table(dbp, &colids);
  if (tp == NULL) LF_ERROR(("Error opening settings table"));

  /* get a row template */
  row = db_alloc_row_template(tp, &ncol);
  
  /* delete any old reference */
  row[colids[LF_TABLE_FMS_SETTINGS_PARAM]].dbd_str = param;
  row[colids[LF_TABLE_FMS_SETTINGS_VALUE]].dd_type = DBD_MATCHALL;
  rc = db_delete_row(tp, row);
  if (rc != 0) LF_ERROR(("Error deleting param from DB"));

  /* now, add the new value */
  row[colids[LF_TABLE_FMS_SETTINGS_VALUE]].dd_type = DBD_STR;
  row[colids[LF_TABLE_FMS_SETTINGS_VALUE]].dbd_str = value;
  rc = db_add_row(tp, row);
  if (rc != 0) LF_ERROR(("Error adding param to DB"));

  /* write the table back to disk */
  rc = db_flush_table(tp);
  if (rc != 0) LF_ERROR(("Error writing settings table"));

  /* close the table and the database */
  db_close_table(tp);
  rc = db_close_database(dbp);
  if (rc != 0) LF_ERROR(("Error closing database"));

  return;

 except:
  exit(1);
}

/*
 * Set the parameter directly with the FMS
 */
static void
set_param_in_fms(
  char *fms_server,
  char *param,
  char *value)
{
}

/*
 * dump settings from FMS
 */
static void
print_all_settings_from_fms()
{
}

/*
 * Dump all settings
 */
static void
print_all_settings_from_db(
  char *fms_run,
  char *db_name)
{
  struct lf_fms_param_def *pp;
  db_database_ptr_t dbp;
  db_table_ptr_t tp;
  db_datum_t criteria;
  db_datum_t **qres;
  char *val;
  int *colids;
  int nres;
  int rc;

  /*
   * open the database table, creating it if it does not exist
   */
  dbp = db_open_database(fms_run, db_name, FALSE);
  if (dbp == NULL) LF_ERROR(("Error opening database"));

  /* open/create the table */
  tp = lf_open_fms_settings_table(dbp, &colids);
  if (tp == NULL) LF_ERROR(("Error opening settings table"));

  /* setup search datum */
  criteria.dd_type = DBD_STR;

  /* do some setup */
  prepare_param_printing();

  /*
   * Scan through parameter table, looking up each param in the DB or
   * printing its default value if not found.
   */
  for (pp=Params; pp->name != NULL; ++pp) {

    /* look for parameter setting */
    criteria.dbd_str = pp->name;
    rc = db_simple_query(tp, colids[LF_TABLE_FMS_SETTINGS_PARAM], &criteria,
			 &qres, &nres);
    if (rc != 0) LF_ERROR(("Error querying database"));
    if (nres > 1) {
      LF_ERROR(("Multiple matches for \"%s\" in database", pp->name));
    }

    if (nres > 0) {
      val = qres[0][colids[LF_TABLE_FMS_SETTINGS_VALUE]].dbd_str;
    } else {
      val = pp->dflt_value;
    }
    print_param(pp, val, nres<1);

    if (nres > 0) {
      db_free_query_res(tp, qres, nres);
    }
  }

  /* close the table and the database */
  db_close_table(tp);
  rc = db_close_database(dbp);
  if (rc != 0) LF_ERROR(("Error closing database"));
  return;

 except:
  exit(1);
}

static void
validate_param(
  char *param,
  char *value)
{
  struct lf_fms_param_def *pp;

  /* Scan through parameter table looking for this param */
  for (pp=Params; pp->name != NULL; ++pp) {
    if (strcmp(pp->name, param) == 0) {
      return;
    }
  }

  /* we couldn't find the parameter */
  fprintf(stderr, "Unknown parameter \"%s\"\n", param);
  exit(1);
}

char *Spaces;
#define PRINT_GAP 4

static void
prepare_param_printing()
{
  struct lf_fms_param_def *pp;
  int max_param_len;

  max_param_len = 0;
  for (pp=Params; pp->name != NULL; ++pp) {
    int l;
    l = strlen(pp->name);
    if (l > max_param_len) max_param_len = l;
  }
  LF_CALLOC(Spaces, char, max_param_len + PRINT_GAP);
  memset(Spaces, ' ', max_param_len + PRINT_GAP);
  return;

 except:
  exit(1);
}

static void
print_param(
  struct lf_fms_param_def *pp,
  char *value,
  int is_dflt)
{
    int l;

    l = strlen(pp->name);
    printf("%s%s", pp->name, Spaces+l);

    /* print units after integer */
    if (pp->type == LF_FPT_INTEGER) {
      printf("%s", value);
      if (atoi(value) == 1) {
	printf(" %s", pp->units_singular);
      } else {
	printf(" %s", pp->units_plural);
      }

    } else if (pp->type == LF_FPT_STRING) {
      if (value == NULL || *value == '\0') {
	printf("<nil>");
      } else {
	printf("\"%s\"", value);
      }
    }

    if (is_dflt) {
      printf(" [default]");
    }
    printf("\n");
}
